package gov.va.med.mhv.usermgmt.persist.ldap;

import netscape.ldap.LDAPAttribute;
import netscape.ldap.LDAPConnection;
import netscape.ldap.LDAPException;
import netscape.ldap.LDAPModification;
import netscape.ldap.LDAPModificationSet;

/**
 * This class models a modification to a named attribute of an LDAPEntry.
 * Since it implements <code>LdapCommand</code>, it can be scheduled for execution
 * against a valid LdapUpdate instance.
 *
 * @author Jon Crater Apr 4, 2006 3:26:06 PM
 */
public abstract class LdapAttributeModification implements LdapCommand {
	private static final int ADD = LDAPModification.ADD;
	private static final int DELETE = LDAPModification.DELETE;
	private static final int REPLACE = LDAPModification.REPLACE;

	/** 
	 * The name of the attribute to which this modification will be applied.
	 */
	private String attributeName;
	
	/**
	 * The new value to be written onto this attribute.
	 */
	private String newValue;
	
	/**
	 * The operation to be performed.  This value corresponds to the the relevant
	 * constant defined in <code>netscape.ldap.LDAPModification</code>
	 * @see netscape.ldap.LDAPModification
	 */
	private int operation = -1;
	
	private LdapAttributeModification(int operation, String attributeName, String newValue) {
		this.operation = operation;
		this.attributeName = attributeName;
		this.newValue = newValue;
	}
	
	private LdapAttributeModification(String attributeName, String newValue) {
		this.attributeName = attributeName;
		this.newValue = newValue;
	}
	
	/**
	 * Factory method for initializing an LdapAttributeModification that encapsulates 
	 * the logic for adding an attribute to an entry.
	 * @param attributeName
	 * @param newValue
	 * @return LdapAttributeModification
	 */
	public static LdapAttributeModification add(String attributeName, String newValue) {
		return new AddModification(attributeName, newValue);
	}
	
	/**
	 * Factory method for initializing an LdapAttributeModification that encapsulates 
	 * the logic for deleting an attribute from an entry.
	 * @param attributeName
	 * @param newValue
	 * @return LdapAttributeModification
	 */
	public static LdapAttributeModification delete(String attributeName) {
		return new DeleteModification(attributeName);
	}
	
	/**
	 * Factory method for initializing an LdapAttributeModification that encapsulates 
	 * the logic for replacing an attribute of an entry.
	 * @param attributeName
	 * @param newValue
	 * @return LdapAttributeModification
	 */
	public static LdapAttributeModification replace(String attributeName, String newValue) {
		return new ReplaceModification(attributeName, newValue);
	}
	
	/**
	 * Factory method for initializing an LdapAttributeModification that encapsulates 
	 * the logic for deleting then adding back an attribute to an entry.  This is necessary
	 * for attributes like passwords, where simply replacing might result in a clear text
	 * password being written to the directory.  When the attribute is deleted then replaced,
	 * the underlying directory attribute behavior is restored.
	 * @param attributeName
	 * @param newValue
	 * @return LdapAttributeModification
	 */
	public static LdapAttributeModification wipeAndReplace(String attributeName, String newValue) {
		return new WipeAndReplaceModification(attributeName, newValue);
	}
	
	/**
	 * Template method to provide common behavior among the various LdapAttributeModification 
	 * classes.
	 * @param distinguishedName
	 * @param conn
	 */
	public void execute(String distinguishedName, LDAPConnection conn) throws LDAPException {
		LDAPModificationSet modSet = new LDAPModificationSet();
		LDAPAttribute att = getAttribute();
		modSet.add(getOperation(), att);
		conn.modify(distinguishedName, modSet);
	}
	
	/**
	 * Return the operation.
	 * @return int
	 */
	public int getOperation() {
		if(operation == -1)
			throw new IllegalStateException("Operation not initialized.");
		
		return operation;
	}
	
	/**
	 * Return the name.
	 * @return String
	 */
	public String getName() {
		return attributeName;
	}
	
	/**
	 * Return the new value that will be set on the attribute.
	 * @return String
	 */
	public String getNewValue() {
		return newValue;
	}
	
	/**
	 * Return an LDAPAttribute on which to operate.  This method satisfies most cases
	 * by simply returning a new instance with the name and value provided.
	 * @return LDAPAttribute
	 */
	protected LDAPAttribute getAttribute() {
		return new LDAPAttribute(getName(), getNewValue());
	}
	
	private static class AddModification extends LdapAttributeModification {
		private AddModification(String attributeName, String newValue) {
			super(ADD, attributeName, newValue);
		}
	}
	
	private static class DeleteModification extends LdapAttributeModification {
		private DeleteModification(String attributeName) {
			super(DELETE, attributeName, null);
		}

		protected LDAPAttribute getAttribute(String distinguishedName, LDAPConnection conn) throws LDAPException {
			return new LDAPAttribute(getName());
		}
	}
	
	private static class ReplaceModification extends LdapAttributeModification {
		private ReplaceModification(String attributeName, String newValue) {
			super(REPLACE, attributeName, newValue);
		}
	}
	
	private static class WipeAndReplaceModification extends LdapAttributeModification {
		private WipeAndReplaceModification(String attributeName, String newValue) {
			super(attributeName, newValue);
		}
		
		public void execute(String distinguishedName, LDAPConnection conn) throws LDAPException {
			new DeleteModification(getName()).execute(distinguishedName, conn);
			new AddModification(getName(), getNewValue()).execute(distinguishedName, conn);
		}
	}
}
